/*:
 * @target MZ
 * @plugindesc [Debug] 武器/防具の所持数が増減したら通知（入手方法問わず） v1.1.0
 * @author ChatGPT
 *
 * @help
 * ■概要
 * 武器・防具の所持数が「増えた/減った」瞬間に、メッセージで通知します。
 * デバッグ用途向けに、入手方法を問わず拾うことを重視しています。
 *
 * ■通知対象
 * ・武器 (Weapons)
 * ・防具 (Armors)
 *
 * ■色指定
 * ・増えた時の色/減った時の色は \C[n] の n（ウィンドウスキンの文字色番号）で指定します。
 * ・例：増加=3, 減少=2 など
 *
 * ■注意
 * ・通知は基本的に $gameParty.gainItem() の呼び出しに追従します。
 * ・所持上限などで実際に変化しなかった場合は通知しません（差分で判定）。
 * ・メッセージ表示は「今メッセージ表示できるシーン（マップ/戦闘など）」で順次出します。
 *
 * @param Enabled
 * @text 有効
 * @type boolean
 * @default true
 *
 * @param EnableSwitchId
 * @text 有効化スイッチ（任意）
 * @type switch
 * @default 0
 * @desc 0なら常にEnabledで判定。1以上なら、そのスイッチON時のみ通知。
 *
 * @param ShowIcon
 * @text アイコンを表示
 * @type boolean
 * @default true
 *
 * @param ShowTotal
 * @text 所持数も表示
 * @type boolean
 * @default true
 *
 * @param Prefix
 * @text 接頭辞
 * @type string
 * @default [DEBUG]
 *
 * @param WeaponLabel
 * @text 武器ラベル
 * @type string
 * @default 武器
 *
 * @param ArmorLabel
 * @text 防具ラベル
 * @type string
 * @default 防具
 *
 * @param GainColorIndex
 * @text 増加色（\C[n] の n）
 * @type number
 * @min 0
 * @default 3
 *
 * @param LoseColorIndex
 * @text 減少色（\C[n] の n）
 * @type number
 * @min 0
 * @default 2
 *
 * @param ColorizeWholeLine
 * @text 行全体を色付け
 * @type boolean
 * @default false
 * @desc ONなら行全体を増減色に。OFFなら「+/-や数量部分」だけ色付け。
 *
 * @command ClearQueue
 * @text 通知キューをクリア
 *
 * @command SetEnabled
 * @text 有効/無効を切替
 * @arg enabled
 * @type boolean
 * @default true
 */

(() => {
  "use strict";

  const PLUGIN_NAME = "GainEquipNoticeDebug";
  const params = PluginManager.parameters(PLUGIN_NAME);

  const P = {
    enabled: params.Enabled === "true",
    enableSwitchId: Number(params.EnableSwitchId || 0),
    showIcon: params.ShowIcon === "true",
    showTotal: params.ShowTotal === "true",
    prefix: String(params.Prefix ?? "[DEBUG]"),
    weaponLabel: String(params.WeaponLabel ?? "武器"),
    armorLabel: String(params.ArmorLabel ?? "防具"),
    gainColorIndex: Number(params.GainColorIndex ?? 3),
    loseColorIndex: Number(params.LoseColorIndex ?? 2),
    colorizeWholeLine: params.ColorizeWholeLine === "true",
  };

  // --- Queue on Game_Temp ---
  const _Game_Temp_initialize = Game_Temp.prototype.initialize;
  Game_Temp.prototype.initialize = function() {
    _Game_Temp_initialize.call(this);
    this._gainEquipNoticeQueue = [];
  };

  function isNoticeEnabled() {
    if (!P.enabled) return false;
    if (P.enableSwitchId > 0) {
      return $gameSwitches && $gameSwitches.value(P.enableSwitchId);
    }
    return true;
  }

  function isEquipItem(item) {
    return item && (DataManager.isWeapon(item) || DataManager.isArmor(item));
  }

  function labelOf(item) {
    return DataManager.isWeapon(item) ? P.weaponLabel : P.armorLabel;
  }

  function iconTextOf(item) {
    if (!P.showIcon) return "";
    const iconIndex = Number(item.iconIndex || 0);
    return iconIndex > 0 ? `\\I[${iconIndex}]` : "";
  }

  function color(codeIndex) {
    return `\\C[${Number(codeIndex)}]`;
  }
  function resetColor() {
    // 0が「通常色」に戻る（ツクールの一般的な扱い）
    return "\\C[0]";
  }

  function buildMessage(item, delta, total) {
    const label = labelOf(item);
    const icon = iconTextOf(item);
    const name = item.name || "";

    const isGain = delta > 0;
    const abs = Math.abs(delta);
    const sign = isGain ? "+" : "-";
    const c = isGain ? color(P.gainColorIndex) : color(P.loseColorIndex);

    // 色付け対象：行全体 or 変化部分だけ
    if (P.colorizeWholeLine) {
      const base = `${P.prefix} ${label} ${icon}${name} ${sign}${abs}`;
      const tail = P.showTotal ? `（所持:${total}）` : "";
      return `${c}${base}${tail}${resetColor()}`;
    } else {
      const changePart = `${c}${sign}${abs}${resetColor()}`;
      const base = `${P.prefix} ${label} ${icon}${name} ${changePart}`;
      const tail = P.showTotal ? `（所持:${total}）` : "";
      return `${base}${tail}`;
    }
  }

  function enqueueNotice(text) {
    if (!$gameTemp) return;
    if (!$gameTemp._gainEquipNoticeQueue) $gameTemp._gainEquipNoticeQueue = [];
    $gameTemp._gainEquipNoticeQueue.push(text);
  }

  function flushNoticeQueue() {
    if (!isNoticeEnabled()) return;
    if (!$gameTemp || !$gameTemp._gainEquipNoticeQueue) return;
    if ($gameTemp._gainEquipNoticeQueue.length <= 0) return;

    // 1フレーム1件（詰まりにくくする）
    if ($gameMessage && !$gameMessage.isBusy()) {
      const text = $gameTemp._gainEquipNoticeQueue.shift();
      $gameMessage.add(text);
    }
  }

  const _Scene_Map_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function() {
    _Scene_Map_update.call(this);
    flushNoticeQueue();
  };

  const _Scene_Battle_update = Scene_Battle.prototype.update;
  Scene_Battle.prototype.update = function() {
    _Scene_Battle_update.call(this);
    flushNoticeQueue();
  };

  // --- Hook gainItem ---
  const _Game_Party_gainItem = Game_Party.prototype.gainItem;
  Game_Party.prototype.gainItem = function(item, amount, includeEquip) {
    const before = isEquipItem(item) ? this.numItems(item) : 0;

    _Game_Party_gainItem.call(this, item, amount, includeEquip);

    if (!isNoticeEnabled()) return;
    if (!isEquipItem(item)) return;

    const after = this.numItems(item);
    const delta = after - before;

    // 増減どちらも通知（0は無視）
    if (delta !== 0) {
      enqueueNotice(buildMessage(item, delta, after));
    }
  };

  // --- Plugin Commands ---
  PluginManager.registerCommand(PLUGIN_NAME, "ClearQueue", () => {
    if ($gameTemp && $gameTemp._gainEquipNoticeQueue) {
      $gameTemp._gainEquipNoticeQueue.length = 0;
    }
  });

  PluginManager.registerCommand(PLUGIN_NAME, "SetEnabled", args => {
    P.enabled = String(args.enabled) === "true";
  });
})();
